// API 설정
const API_BASE_URL = "https://hmpc.uk";

// DOM 요소
const shortenForm = document.getElementById("shortenForm");
const customIdInput = document.getElementById("customId");
const memoInput = document.getElementById("memo");
const statusMessage = document.getElementById("statusMessage");
const pageTitleElement = document.getElementById("pageTitle");
const pageUrlElement = document.getElementById("pageUrl");
const resultSection = document.getElementById("resultSection");
const shortUrlText = document.getElementById("shortUrlText");
const submitBtn = document.getElementById("submitBtn");

// 현재 탭 정보 저장
let currentTab = null;

// 초기화
document.addEventListener("DOMContentLoaded", async () => {
  await loadCurrentTab();
  shortenForm.addEventListener("submit", handleSubmit);
});

// 현재 탭 정보 로드
async function loadCurrentTab() {
  try {
    const [tab] = await chrome.tabs.query({
      active: true,
      currentWindow: true,
    });
    currentTab = tab;

    if (tab) {
      pageTitleElement.textContent = tab.title || "Untitled";
      pageUrlElement.textContent = tab.url || "";
    }
  } catch (error) {
    console.error("Error loading tab info:", error);
    pageTitleElement.textContent = "페이지 정보를 불러올 수 없습니다";
    showMessage("페이지 정보를 가져오는데 실패했습니다.", "error");
  }
}

// 폼 제출 핸들러
async function handleSubmit(e) {
  e.preventDefault();

  if (!currentTab || !currentTab.url) {
    showMessage("현재 페이지 정보를 찾을 수 없습니다.", "error");
    return;
  }

  const destination = currentTab.url;
  const customId = customIdInput.value.trim();
  const memo = memoInput.value.trim();
  const title = currentTab.title;

  // 버튼 비활성화
  submitBtn.disabled = true;
  submitBtn.textContent = "생성 중...";

  try {
    const requestBody = {
      destination,
      title: title || "Untitled",
    };

    if (customId) {
      requestBody.id = customId;
    }

    if (memo) {
      requestBody.memo = memo;
    }

    const response = await fetch(`${API_BASE_URL}/api/shorten`, {
      method: "POST",
      headers: {
        "Content-Type": "application/json",
      },
      body: JSON.stringify(requestBody),
    });

    if (!response.ok) {
      const error = await response.json();
      throw new Error(error.error || "오류가 발생했습니다.");
    }

    const data = await response.json();

    // 성공 메시지 표시
    showMessage("URL이 성공적으로 단축되었습니다!", "success");

    // 결과 표시 및 자동 복사
    await showResult(data.shortUrl);

    // 폼 초기화
    shortenForm.reset();
  } catch (error) {
    showMessage(error.message, "error");
  } finally {
    // 버튼 활성화
    submitBtn.disabled = false;
    submitBtn.innerHTML = `
            <svg class="icon" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <path d="M10 13a5 5 0 0 0 7.54.54l3-3a5 5 0 0 0-7.07-7.07l-1.72 1.71"></path>
                <path d="M14 11a5 5 0 0 0-7.54-.54l-3 3a5 5 0 0 0 7.07 7.07l1.71-1.71"></path>
            </svg>
            단축하고 복사하기
        `;
  }
}

// 결과 표시 및 자동 복사
async function showResult(url) {
  shortUrlText.value = url;
  resultSection.style.display = "block";

  // 자동으로 클립보드에 복사
  try {
    await navigator.clipboard.writeText(url);
  } catch (error) {
    console.error("Auto copy failed:", error);
  }
}

// 클립보드에 복사
async function copyToClipboard() {
  const url = shortUrlText.value;

  try {
    await navigator.clipboard.writeText(url);

    const copyBtn = document.querySelector(".copy-btn");
    const originalHtml = copyBtn.innerHTML;

    copyBtn.innerHTML = `
            <svg class="icon" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <polyline points="20 6 9 17 4 12"></polyline>
            </svg>
            복사됨!
        `;

    setTimeout(() => {
      copyBtn.innerHTML = originalHtml;
    }, 2000);

    showMessage("클립보드에 복사되었습니다!", "success");
  } catch (error) {
    showMessage("복사에 실패했습니다.", "error");
  }
}

// 상태 메시지 표시
function showMessage(message, type = "success") {
  statusMessage.textContent = message;
  statusMessage.className = `status-message ${type} show`;

  setTimeout(() => {
    statusMessage.classList.remove("show");
  }, 5000);
}
